/*____________________________________________________________________________
		Copyright (C) 2000 Network Associates, Inc.
        All rights reserved.

        $Id: CSecureString.cpp,v 1.3 1999/10/27 07:28:41 nryan Exp $
____________________________________________________________________________*/

#include "pgpClassesConfig.h"
#include <stdio.h>

#include "UTime.h"
#include "CSecureString.h"

_USING_PGP

// Types

struct CSecureString::SecureChar : public CListableObject<SecureChar>
{
	PGPByte	obfuChar;
	PGPByte	key;
};


// Class CSecureString member functions

CSecureString::CSecureString(const char *str)
{
#if PGP_EXCEPTIONS
	Assign(str);
#else	// !PGP_EXCEPTIONS
	Status() = Assign(str);
#endif	// PGP_EXCEPTIONS
}

CSecureString::CSecureString(const CSecureString& csString)
{
#if PGP_EXCEPTIONS
	Assign(csString);
#else	// !PGP_EXCEPTIONS
	Status() = Assign(csString);
#endif	// PGP_EXCEPTIONS
}

CSecureString::~CSecureString()
{
	Clear();
}

PGPBoolean 
CSecureString::operator==(const CSecureString& csString) const
{
	if (&csString == this)
		return TRUE;

	if (Length() != csString.Length())
		return FALSE;

	for (PGPUInt32 i = 0; i < Length(); i++)
	{
		if (GetAt(i) != csString.GetAt(i))
			return FALSE;
	}

	return TRUE;
}

char 
CSecureString::GetAt(PGPInt32 pos) const
{
	SecureChar	*pChar	= GetSCAtIndex(pos);
	return (pChar->obfuChar ^ pChar->key);
}

void 
CSecureString::RemoveAt(PGPInt32 pos)
{
	SecureChar	*pChar	= GetSCAtIndex(pos);
	mCharList.Remove(pChar);
}

void 
CSecureString::SetAt(PGPInt32 pos, char c)
{
	SecureChar	*pChar	= GetSCAtIndex(pos);

	pChar->key = MakeNewKey();
	pChar->obfuChar = (c ^ pChar->key);
}

SMART_ERROR 
CSecureString::InsertAfter(PGPInt32 pos, char c)
{
	SMART_ERROR_DECLARE

	SecureChar	*pNewChar	= new SecureChar;

#if !PGP_EXCEPTIONS

	if (IsNull(pNewChar))
		error.pgpErr = kPGPError_OutOfMemory;

	if (error.IsntError())
#endif	// !PGP_EXCEPTIONS
	{
		pNewChar->key = MakeNewKey();
		pNewChar->obfuChar = (c ^ pNewChar->key);

		if (pos == -1)
			mCharList.AddHead(pNewChar);
		else
			mCharList.InsertAfter(GetSCAtIndex(pos), pNewChar);
	}

	SMART_ERROR_RETURN
}

void 
CSecureString::Extract(CSecureArray<char>& secureChars) const
{
	secureChars.Resize(Length() + 1);

	for (PGPUInt32 i = 0; i < Length(); i++)
		secureChars[i] = GetAt(i);

	secureChars[i] = '\0';
}

void 
CSecureString::Clear()
{
	mCharList.EmptyWithDelete();
}

SMART_ERROR 
CSecureString::Assign(const char *str)
{
	SMART_ERROR_DECLARE
	Clear();

	PGPInt32	length	= strlen(str);

	for (PGPInt32 i = 0; i < length; i++)
	{
	#if PGP_EXCEPTIONS

		try
		{
			InsertAfter(i - 1, str[i]);
		}
		catch (CComboError&)
		{
			Clear();
			throw;
		}

	#else	// !PGP_EXCEPTIONS

		error = InsertAfter(i - 1, str[i]);

		if (error.IsError())
		{
			Clear();
			break;
		}

	#endif	// !PGP_EXCEPTIONS
	}

	SMART_ERROR_RETURN
}

SMART_ERROR 
CSecureString::Assign(const CSecureString& csString)
{
	SMART_ERROR_DECLARE
	Clear();

	PGPInt32	length	= csString.Length();

	for (PGPInt32 i = 0; i < length; i++)
	{
	#if PGP_EXCEPTIONS

		try
		{
			InsertAfter(i - 1, csString[i]);
		}
		catch (CComboError&)
		{
			Clear();
			throw;
		}

	#else	// !PGP_EXCEPTIONS

		error = InsertAfter(i - 1, csString[i]);

		if (error.IsError())
		{
			Clear();
			break;
		}

	#endif	// !PGP_EXCEPTIONS
	}

	SMART_ERROR_RETURN
}

CSecureString::SecureChar * 
CSecureString::GetSCAtIndex(PGPInt32 index) const
{
	pgpAssert((index >= 0) && (index <= static_cast<PGPInt32>(Length())));

	PGPInt32	i		= 0;
	SecureChar	*pChar	= mCharList.Head();

	while (i++ < index)
	{
		pChar = mCharList.Next(pChar);
	}

	return pChar;
}

PGPByte 
CSecureString::MakeNewKey() const
{
	return static_cast<PGPByte>(UTime::GetSystemTicks() % 0xFF);
}
